import fs from 'fs';
import path from 'path';

/**
 * Clase para gestionar el registro de logs.
 * 
 * @class
**/
class Logger {
	/**
	 * Crea una instancia de Logger.
	 * 
	 * @constructor
	**/
	constructor() {
		/**
		 * Directorio donde se almacenarán los archivos de log.
		 * 
		 * @member {string}
		**/
		this.logDirectory = 'logs';

		// Verifica la existencia del directorio, de no existir crea el directorio "logs".
		if ( !fs.existsSync( this.logDirectory ) ) fs.mkdirSync( this.logDirectory );
	}

	/**
	 * Genera la fecha actual en un formato específico.
	 * 
	 * @returns {string} Fecha en formato dd/mm/yyyy HH:mm:ss.
	**/
	generarFecha() {
		const date = new Date();
		const options = {
			timeZone: 'America/Mexico_City',
			year: 'numeric',
			month: '2-digit',
			day: '2-digit',
			hour: '2-digit',
			minute: '2-digit',
			second: '2-digit',
			hour12: false
		};
		const time = date.toLocaleString('es-MX', options);
		return time;
	}

	/**
	 * Escribe un registro en el archivo de log.
	 * 
	 * @param {string} tipo - El tipo de mensaje (Error, Info, etc.).
	 * @param {string} mensaje - El mensaje a registrar en el log.
	**/
	escribirLog( tipo, mensaje ) {
		/**
		 * Tipo de log permitidos.
		 * @type {string}
		**/
		const tiposPermitidos = ['error', 'info', 'debug'];

		//
		tipo = tipo.toLowerCase();

		// Validar que el tipo de log recibido sea correcto
		if ( !tiposPermitidos.includes( tipo ) ) {
			throw new Error( `Tipo de log no válido: ${tipo}` );
		}
		else {
			/**
			 * Nombre del archivo de log.
			 * @type {string}
			**/
			const logFileName = `${tipo}.log`;

			/**
			 * Ruta completa del archivo de log.
			 * @type {string}
			**/
			const logPath = path.join(this.logDirectory, logFileName);

			/**
			 * Entrada de log a registrar.
			 * @type {string}
			 */
			const logEntry = `[${this.generarFecha()}] ${mensaje}\n`;

			fs.appendFile( logPath, logEntry, 'utf-8', ( error ) => {
				if ( error ) {
					throw new Error( `Error al escribir en el archivo de log: ${error}` );
				}
			});
		}
	}
}

// Exportar la instancia del logger
export default Logger;